/*
Welcome to Project Extensions! Place code in this file or in files contained in this directory to add customized Incisor functionality to your project.
Keep in mind that code placed in this 'ProjectExtensions' directory will not be included in published projects, and is instead meant
to provide a space for scripts and functionality to help in the development process. You can add custom menus, startup scripts, pre/post publish 
scripts, define custom inspector panels, and more. 

Please note that this functionality is only available with certain Incisor subscriptions, so check your account settings to ensure that 'extensions' is 
included among the detailed permissions.

The examples below outline some of the key ways that Incisor extensions can help developers enhance productivity for their process and their team.

The 'nce' var is the primary accessor to Incisor extensions functionality, and largely mirrors the functionality that the 'nc' object has within normal project code.
*/




// SAMPLE PROJECT EXTENSIONS CODE:
/*


// Classes can be defined within extensions for object-oriented organization...
class MyExtensionsFunctionalityClass
{
    constructor(){}


    showGreetingsAlert()
    {
        nce.alert("Greetings","Greetings! This alert has been brought to you by Incisor extensions...");
    }


    showSubMenuGreetingsAlert()
    {
        nce.alert("Greetings from a Sub-Menu","Greetings! This alert has been brought to you by Incisor extensions from within a sub-menu...");
    }
}


// Custom InspectorPanels can be defined for use within the Incisor GUI. Just define a class that extends InspectorPanel, then call 'nce.registerInspectorPanelDefinition'.
class SampleCustomPanel extends InspectorPanel
{
    constructor()
    {
        super("Sample Custom Panel");

        // Add a sample title to the header ScrollingPanel
        this.titleText = new UiText(this,"TitleText");
        this.titleText.string = "Sample Custom Panel:";
        this.titleText.textFormat = nce.uiStyle.textFormat_subHeading.clone();
        this.titleText.layoutObject.topBuffer = nce.uiStyle.spacer_small;
        this.titleText.layoutObject.leftBuffer = nce.uiStyle.spacer_small*2;
        this.headerScrollingPanel.bodyStack.addElements(this.titleText);

        // Add a collapsible stack to the InspectorPanel body 
        this.standardPopupsCollapsibleStack = new UiCollapsibleStack(this,"standardPopupsCollapsibleStack");
        this.standardPopupsCollapsibleStack.layoutObject.topBuffer = nce.uiStyle.spacer_small;
        this.standardPopupsCollapsibleStack.titleTextBox.string = "Standard Popups";
        this.bodyScrollingPanel.bodyStack.elementSpacer = nce.uiStyle.spacer_medium;
        this.bodyScrollingPanel.bodyStack.addElements(this.standardPopupsCollapsibleStack);

        // Add a button showing how to use a standard alert popup
        this.standardAlertPopupButton = new UiButton(this,"standardAlertPopupButton");
        this.standardAlertPopupButton.addTriggerCallback(this,"showStandardAlertPopup");
        this.standardAlertPopupButton.label.string = "Standard Alert: ";
        this.standardAlertPopupButton.mainText.string = "Show";
        this.standardPopupsCollapsibleStack.bodyStack.addElements(this.standardAlertPopupButton);

        // Add a button showing how to use a standard yes no query popup
        this.yesNoQueryPopupButton = new UiButton(this,"yesNoQueryPopupButton");
        this.yesNoQueryPopupButton.addTriggerCallback(this,"showYesNoQueryPopup");
        this.yesNoQueryPopupButton.label.string = "Yes/No Query: ";
        this.yesNoQueryPopupButton.mainText.string = "Show";
        this.standardPopupsCollapsibleStack.bodyStack.addElements(this.yesNoQueryPopupButton);

        // Add a button showing how to use a standard text query popup
        this.textQueryPopupButton = new UiButton(this,"textQueryPopupButton");
        this.textQueryPopupButton.addTriggerCallback(this,"showTextQueryPopup");
        this.textQueryPopupButton.label.string = "Text Query: ";
        this.textQueryPopupButton.mainText.string = "Show";
        this.standardPopupsCollapsibleStack.bodyStack.addElements(this.textQueryPopupButton);
        
        // Add a button showing how to use a custom popup
        this.customPopupButton = new UiButton(this,"customPopupButton");
        this.customPopupButton.addTriggerCallback(this,"showCustomPopup");
        this.customPopupButton.label.string = "Custom Popup: ";
        this.customPopupButton.mainText.string = "Show";
        this.standardPopupsCollapsibleStack.bodyStack.addElements(this.customPopupButton);

        // Add a drop down menu with screen shot options
        this.screenshotOptions = new UiDropDownMenu(this,"screenshotOptions");
        this.screenshotOptions.label.string = "Screen Shot:";
        this.screenshotOptions.mainText.string = "Options...";
        this.screenshotOptions.uiMenu.addMenuItem("Save Project Screenshot");
        this.screenshotOptions.uiMenu.addMenuItem("Save Inspector Screenshot");
        this.screenshotOptions.layoutObject.leftBuffer = nce.uiStyle.spacer_medium;
        this.screenshotOptions.uiMenu.searchable = false;
        this.screenshotOptions.uiMenu.addMenuItemTriggerCallback(this,"screenShotsMenuItemCallback");
        this.bodyScrollingPanel.bodyStack.addElements(this.screenshotOptions);

        // Add button to write all SceneObject names to text file
        this.writeAllSceneObjectNamesButton = new UiButton(this,"writeAllSceneObjectNamesButton");
        this.writeAllSceneObjectNamesButton.label.string = "List All SceneObjects in Scene:";
        this.writeAllSceneObjectNamesButton.mainText.string = "Export to File";
        this.writeAllSceneObjectNamesButton.layoutObject.leftBuffer = nce.uiStyle.spacer_medium;
        this.writeAllSceneObjectNamesButton.addTriggerCallback(this,"writeAllSceneObjectNamesToTextFile");
        this.bodyScrollingPanel.bodyStack.addElements(this.writeAllSceneObjectNamesButton);

        // Add button to show progress bars
        this.showProgressBarsButton = new UiGraphicButton(this,"showProgressBarsButton");
        this.showProgressBarsButton.label.string = "Show Progress Bars:";
        this.showProgressBarsButton.graphicObject.graphicAsset = nce.graphicAssets.incr_icon_horizarrow;
        this.showProgressBarsButton.layoutObject.leftBuffer = nce.uiStyle.spacer_medium;
        this.showProgressBarsButton.addTriggerCallback(this,"showProgressBars");
        this.bodyScrollingPanel.bodyStack.addElements(this.showProgressBarsButton);
    }


    showProgressBars()
    {
        nce.showProgressBar("First Task",.25,"Getting started...");
        nce.waitThen(1,nce,"showProgressBar",["First Task",.55,"Looking good..."]);
        nce.waitThen(1.5,nce,"showProgressBar",["First Task",.75,"Almost done..."]);
        nce.waitThen(2.5,nce,"showProgressBar",["First Task",1]);
        nce.waitThen(1.2,nce,"showProgressBar",["Second Task",.35,"Getting started..."]);
        nce.waitThen(1.6,nce,"showProgressBar",["Second Task",.4,"Task in progress..."]);
        nce.waitThen(3,nce,"showProgressBar",["Second Task",.7,"Getting closer..."]);
        nce.waitThen(4,nce,"showProgressBar",["Second Task",.8,"Almost done..."]);
        nce.waitThen(4.2,nce,"showProgressBar",["Second Task",1]);
    }


    showStandardAlertPopup()
    {
        nce.alert("Sample Alert","This is a sample alert!");
    }


    async showYesNoQueryPopup()
    {
        let response = await nce.showYesNoQueryPopup("Query","Would you like to test this popup?");
        nce.alert("User Response","The user response was: " + response);
    }


    async showTextQueryPopup()
    {
        let response = await nce.showTextQueryPopup("Query","What's your favorite color?","color:","Red");
        nce.alert("Favorite Color","The favorite color was: " + response);
    }


    async showCustomPopup()
    {
        if(this.customPopupInstance===undefined)
        {
            // Make a new UiPopupWindow if it doesn't already exist
            this.customPopupInstance = new UiPopupWindow(nce.mainScene,"CustomPopup");

            // Set the popup title text string
            this.customPopupInstance.titleText.string = "Custom Job";
            this.customPopupInstance.titleText.textFormat = nce.uiStyle.textFormat_heading;
            this.customPopupInstance.titleText.colorMultiply.copy(nce.uiStyle.color_interactiveText);

            // Set the 'decision' buttons
            this.customPopupInstance.setDecisionButtons(["These", "Answers", "Are", "Custom"]);

            // Add a callback for the decision buttons
            this.customPopupInstance.addDecisionButtonTriggerCallback(this,"closeCustomPopup");

            // Add some body text
            let bodyText = new UiText(this,"bodyText");
            bodyText.string = "Populate anything in the 'bodyStack' LayoutStack...";
            this.customPopupInstance.bodyStack.addElements(bodyText);

            // Add a graphic
            let movingGraphic = new GraphicObject(nce.graphicAssets.incr_icon_incisorlogo,this);
            movingGraphic.scale.addMotion.all(.95,1.05,1);
            this.customPopupInstance.bodyStack.addElements(movingGraphic);

            // Add a CollapsibleStack
            let collapsibleStack = new UiCollapsibleStack(this,"collapsibleStack");
            collapsibleStack.titleTextBox.string = "Open Me...";
            this.customPopupInstance.bodyStack.addElements(collapsibleStack);

            // Add a UiTextField
            let textField = new UiTextField(this,"textField");
            textField.label.string = "Test Text Field:";
            textField.mainText.string = "Write something here..";
            collapsibleStack.bodyStack.addElements(textField);
        }
        this.customPopupInstance.isOpen = true;
        await this.customPopupInstance.awaitClose();
    }


    closeCustomPopup(decision)
    {
        this.customPopupInstance.isOpen = false;
        nce.alert("Custom Popup Response","The response to the custom popup was: '" + decision + "'" );
    }


    async screenShotsMenuItemCallback(uiMenuItem,uiMenu)
    {
        if(uiMenuItem.name==="Save Project Screenshot")
        {
            let path = await nce.saveProjectScreenShot();
            nce.alert("Screen Shot Saved", "A screen shot of the project has been saved to this location: \n" + path);
        }
        else if(uiMenuItem.name==="Save Inspector Screenshot")
        {
            let path = await nce.saveInspectorScreenShot();
            nce.alert("Screen Shot Saved", "A screen shot of the inspector has been saved to this location: \n" + path);
        }
        uiMenu.isOpen = false;
    }


    writeAllSceneObjectNamesToTextFile()
    {
    // 'nce.projectNC' is the means to access the 'nc' object associated with the project from extensions scripts...
        if(nce.projectNC!==undefined)
        {
            let descendants = nce.projectNC.mainScene.getDescendants();
            let fileText = "";
            for(let i=0; i<descendants.length; i++)
            {
                fileText+=descendants[i].name+":"+descendants[i].type+"\n";
            }
            nce.fileIO.writeTextFile("AllSceneObjectNames.txt",fileText);
        }
    // 'nce.projectNC' can return undefined when the project is not connected; this can happen when the project is loading or has had a runtime error.
        else
        {
            nce.alert("Can't Access Project","The Incisor Inspector is not currently connected to the project, and as a result the SceneObjects cannont be listed.");
        }
    }

}


// Place code that should run in the 'init' phase within 'CODE.runBeforeInit' blocks. This scope will have access to 'nce', 
// and all user-created classes will be defined before this code executes.
CODE.runBeforeInit___begin;
    console.log("Project Extensions Scope - This code executes when the Inspector first opens, after any global extensions code.");

    // Instantiate MyExtensionsFunctionalityClass
    var myExtensionsFunctionality = new MyExtensionsFunctionalityClass();

    // Add a new custom menu item that shows a standard alert popup...
    nce.addInspectorMenuItem(["Sample Extension Menu", "Greetings Popup"],myExtensionsFunctionality,"showGreetingsAlert");
    // Add a new custom menu item in a sub-menu that shows a standard alert popup...
    nce.addInspectorMenuItem(["Sample Extension Menu","Sub-Menu Example", "Sub-Menu Greetings Popup"],myExtensionsFunctionality,"showSubMenuGreetingsAlert");

    // Register a the custom InspectorPanel 'SampleCustomPanel' with Incisor.
    nce.registerInspectorPanelDefinition("Sample Custom Panel",SampleCustomPanel);   
CODE.runBeforeInit___end;

*/